/*--------------------------------------------------------------------
 * FICHERO:  Comprime.c
 * OBJETIVO: Comprimir una lista de ficheros en uno slo
 * AUTOR:    Pedro Reina
 * FECHA:    M.21.7.1998
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include "ComprExp.h"

/*--------------------------------------------------------------------
 * Definicin de mensajes al usuario
 *------------------------------------------------------------------*/

#ifdef PR_CASTELLANO
#define FicheroDatos        "\nFichero de datos: %s\n"
#define NumeroDeFicheros    "\nNmero de ficheros: %d\n"
#define ComprimePedroReina  "Comprime, v 2.1, Pedro Reina, Julio 1998"
#define DemasiadosFicheros  "Demasiados ficheros"
#define Uso "\nPara invocar el programa:"                             \
            "\nComprime NombreFichero"                                \
            "\n  donde NombreFichero es el nombre de un fichero que"  \
            "\n  tiene extensin 'lst' y contiene la relacin de"     \
            "\n  ficheros que hay que comprimir\n"
#endif

#ifdef PR_INGLES
#define FicheroDatos        "\nData file: %s\n"
#define NumeroDeFicheros    "\nNumber of files: %d\n"
#define ComprimePedroReina  "Comprime, v 2.1 English, Pedro Reina, July 1998"
#define DemasiadosFicheros  "Too many files"
#define Uso "\nTo call this programme:"                               \
            "\nComprime FileName"                                     \
            "\n  where FileName is the name of a file with extension" \
            "\n  'lst' that holds the relation of files to compress\n"
#endif

/*--------------------------------------------------------------------
 * Ampliamos la pila del QL
 *------------------------------------------------------------------*/

#ifdef PR_QDOS
long _stack = 8000;
#endif

/*--------------------------------------------------------------------
 * Definicin de macros constantes
 *------------------------------------------------------------------*/

 /* Ninguno */

/*--------------------------------------------------------------------
 * Definicin de variables globales
 *------------------------------------------------------------------*/

/* Compresin LZSS */
int short Superior[VENTANA+1], Derecho[VENTANA+257], Izquierdo[VENTANA+1];
int short LongIgual, PosIgual;

/*--------------------------------------------------------------------
 * Declaracin de funciones
 *------------------------------------------------------------------*/

/* Las funciones que presentan el programa */
void Presenta (void);
void MuestraUso (void);

/* Las funciones que leen la lista de ficheros */
short int ExtraeNombres (char *, char[][], char[]);
char LeePalabra (FILE *, char *, char *);

/* Direccin general de la compresin */
void Comprime (char *, char);
void EscribeFalsaCabecera (void);
void EscribeCabecera (char *, char, char, unsigned long, unsigned long);
char LeeOctetoPlano (unsigned char *);
void EscribeOctetoComprimido (unsigned char);

/* Compresin plana */
void ComprimePlano (void);

/* Compresin LZH */
char ComprimeLZH (void);
void IniciaArbol (void);
void InsertaNodo (int short);
void BorraNodo (int short);
void EscribeBit (int short, unsigned short);
void CodificaOcteto (unsigned short);
void CodificaPosicion (unsigned short);
void CodificaFinal (void);

/*--------------------------------------------------------------------
 * Definicin de tablas de consulta para codificar los 6 bits
 *   superiores de los smbolos
 *------------------------------------------------------------------*/

unsigned char ValorLong[64] =
  {
  0x03, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05,
  0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x06,
  0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
  0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
  0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
  0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
  0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
  0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
  };

unsigned char ValorCodigo[64] =
  {
  0x00, 0x20, 0x30, 0x40, 0x50, 0x58, 0x60, 0x68,
  0x70, 0x78, 0x80, 0x88, 0x90, 0x94, 0x98, 0x9C,
  0xA0, 0xA4, 0xA8, 0xAC, 0xB0, 0xB4, 0xB8, 0xBC,
  0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE,
  0xD0, 0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE,
  0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
  0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
  0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
  };

/*--------------------------------------------------------------------
 * Definicin de macros funciones
 *------------------------------------------------------------------*/

 /* Ninguno */

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  main()
 * OBJETIVO: Ensamblar las distintas partes del programa
 * ENTRADAS: El nmero de parmetros y los parmetros. El parmetro 1
 *           debe ser el nombre sin extensin del fichero que contiene
 *           la lista de ficheros que hay que comprimir
 * SALIDAS:  Si todo va bien, se crea el fichero con el resultado de la
 *           compresin
 * EJEMPLO:  El programa se invoca as: Comprime Trabajo
 *------------------------------------------------------------------*/
void main (int Narg, char *Arg[])
  {
  char NombreLista[LONG_NOMBRE+1], NombreSalida[LONG_NOMBRE+1];
  char Mensaje[LONG_MENSAJE];
  char ListaNombres[MAX_FICHERO][LONG_NOMBRE+1];
  char ListaTipos[MAX_FICHERO];
  int short TotalFichero, i;

  /* Preparacin */
  setbuf (stdout, NULL);
  Presenta();
  if ( Narg != 2 )
    { MuestraUso(); }

  /* Averiguamos el nombre del fichero con la lista */
  if ( strlen (Arg[1]) > LONG_NOMBRE-4 )
    {
    sprintf (Mensaje, NombreDemasiadoLargo, Arg[1]);
    ErrorFatal (Mensaje);
    }
  strcpy (NombreLista, Arg[1]);
  strcat (NombreLista, ".lst");
  ArreglaSeparador (NombreLista);

  /* Rellenamos la lista con los nombres de los ficheros */
  printf (FicheroDatos, NombreLista);
  TotalFichero = ExtraeNombres (NombreLista, ListaNombres, ListaTipos);
  printf (NumeroDeFicheros, TotalFichero);

  /* Preparamos el nombre del fichero destino */
  strcpy (NombreSalida, Arg[1]);
  strcat (NombreSalida, ".cmp");
  ArreglaSeparador (NombreSalida);

  /* Abrimos el fichero donde vamos a dejar los datos comprimidos */
  Salida = fopen (NombreSalida,"wb+");
  if ( !Salida )
    {
    sprintf (Mensaje, ElFicheroNoSeCrea, NombreSalida);
    ErrorFatal (Mensaje);
    }

  /* Realizamos la compresin de cada fichero */
  for ( i=0 ; i<TotalFichero ; i++ )
    { Comprime (ListaNombres[i], ListaTipos[i]); }

  /* Hemos acabado */
  EscribeFalsaCabecera();
  fclose (Salida);
  printf (Terminado);
  }

/*--------------------------------------------------------------------
 * FUNCION:  Presenta()
 * OBJETIVO: Presentar el programa
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Presenta (void)
  {
  short int i;

  #ifdef PR_LINUX_MSDOS
    printf ("\n");
  #endif
  printf (ComprimePedroReina);
  putchar ('\n');
  for ( i=0 ; i<strlen(ComprimePedroReina) ; i++ )
    { putchar ('='); }
  putchar ('\n');
  }

/*--------------------------------------------------------------------
 * FUNCION:  MuestraUso()
 * OBJETIVO: Decir cmo se invoca correctamente el programa
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna. El programa aborta
 *------------------------------------------------------------------*/
void MuestraUso (void)
  {
  printf (Uso);
  exit (0);
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExtraeNombres()
 * OBJETIVO: Extraer de un fichero una lista de nombres y tipos
 * ENTRADAS: El nombre del fichero
 * SALIDAS:  El nmero de nombres encontrados
 *------------------------------------------------------------------*/
short int ExtraeNombres (char NombreFichero[],
                         char ListaNombres[][LONG_NOMBRE+1],
                         char ListaTipos[])
  {
  FILE *Lista;
  char Mensaje[LONG_MENSAJE], Nombre[LONG_MENSAJE], Tipo;
  short int Respuesta;

  /* Abrimos el fichero con la lista */
  Lista = fopen (NombreFichero,"r");
  if ( !Lista )
    {
    sprintf (Mensaje, ElFicheroNoSeAbre, NombreFichero);
    ErrorFatal (Mensaje);
    }

  /* Leemos los nombres que contiene */
  for ( Respuesta=0 ; LeePalabra (Lista, Nombre, &Tipo) ; Respuesta++ )
    {
    if ( Respuesta > MAX_FICHERO )
      { ErrorFatal (DemasiadosFicheros); }
    if ( strlen (Nombre) > LONG_NOMBRE )
      {
      sprintf (Mensaje, NombreDemasiadoLargo, Nombre);
      ErrorFatal (Mensaje);
      }
    ArreglaSeparador (Nombre);
    strcpy (ListaNombres[Respuesta],Nombre);
    ListaTipos[Respuesta] = Tipo;
    }

  /* Recogemos y volvemos */
  fclose (Lista);
  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  LeePalabra()
 * OBJETIVO: Leer la siguiente palabra de un fichero
 * ENTRADAS: El fichero y un lugar donde dejar la palabra y el tipo
 * SALIDAS:  Lgica, que indica si se ha podido leer.
 *           La palabra y el tipo pueden quedar modificadas.
 * NOTA:     Se lee la primera palabra de cada lnea que no comience
 *           con un asterisco ni est en blanco
 *------------------------------------------------------------------*/
char LeePalabra (FILE *Fichero, char *Palabra, char *Tipo)
  {
  char Respuesta, Sigue=1, *Aux;
  char Linea [LONG_MENSAJE];
  
  /* El tipo de fichero por defecto es TEXTO */
  #ifdef PR_LINUX
    *Tipo = TEXTO_LINUX;
  #endif
  #ifdef PR_MSDOS
    *Tipo = TEXTO_MSDOS;
  #endif
  #ifdef PR_QDOS
    *Tipo = TEXTO_QDOS;
  #endif

  while ( Sigue )
    {
    if ( fgets (Linea, LONG_MENSAJE, Fichero) )
      {
      if ( Linea[0] != '\n' && Linea[0] != '*' )
        {
        if ( Aux = strtok (Linea, " \n\t") )
          {
          strcpy (Palabra,Aux);
          Sigue = 0;
          Respuesta = 1;
          if ( Aux = strtok (NULL, " \t") )
            {
            if ( toupper(Aux[0]) == 'B' )  { *Tipo = BINARIO; }
            }
          }
        }
      }
    else
      {
      Sigue = 0;
      Respuesta = 0;
      }
    }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Comprime()
 * OBJETIVO: Dirigir la compresin de un fichero
 * ENTRADAS: El nombre y el tipo del fichero
 * SALIDAS:  Ninguna, pero el programa puede abortar
 *------------------------------------------------------------------*/
void Comprime (char Nombre[], char Tipo)
  {
  short int Ratio, i;
  char Mensaje[LONG_MENSAJE];
  unsigned long PosicionInicial, PosicionFinal;
  char Metodo, Exito;

  /* Abrimos el fichero */
  Entrada = fopen (Nombre, "rb");
  if ( !Entrada )
    {
    sprintf (Mensaje, ElFicheroNoSeAbre, Nombre);
    ErrorFatal (Mensaje);
    }

  /* Informamos al usuario */
  printf ("\n%-20s", Nombre);
  for ( i=0 ; i<PUNTOINDICA ; i++)
    { putchar ('.');  }
  for ( i=0 ; i<PUNTOINDICA ; i++)
    { CursorAtras(); }

  /* Anotamos la posicin en que empezaremos a grabar */
  PosicionInicial = ftell (Salida);
  EscribeFalsaCabecera();

  /* Calculamos la longitud del fichero */
  fseek (Entrada, 0, SEEK_END);
  TamanoOriginal = ftell (Entrada);
  rewind (Entrada);

  /* Preparamos las estadsticas */
  Paso = TamanoOriginal / PUNTOINDICA;

  /* Intentamos comprimir con el mtodo LZH */
  ContadorParcial = 0;
  TamanoComprimido = 0;
  Punto = 0;
  Metodo = CMP_LZH;
  Exito = ComprimeLZH();
  for ( i=Punto ; i<PUNTOINDICA ; i++)
    { putchar ('o');  }

  /* Si no hemos podido con LZH, hay que almacenar el fichero tal cual */
  if ( !Exito )
    {
    rewind (Entrada);
    fseek (Salida, PosicionInicial+CABECERA, SEEK_SET);
    for ( i=0 ; i<PUNTOINDICA ; i++)
      { CursorAtras(); }
    for ( i=0 ; i<PUNTOINDICA ; i++)
      { putchar ('.');  }
    for ( i=0 ; i<PUNTOINDICA ; i++)
      { CursorAtras(); }
    ContadorParcial = 0;
    TamanoComprimido = 0;
    Punto = 0;
    Metodo = CMP_PLANO;
    ComprimePlano();
    for ( i=Punto ; i<PUNTOINDICA ; i++)
      { putchar ('o');  }
    }

  /* Anotamos en su sitio la cabecera correcta y cerramos */
  PosicionFinal = ftell (Salida);
  fseek (Salida, PosicionInicial, SEEK_SET);
  EscribeCabecera (Nombre, Tipo, Metodo, TamanoOriginal, TamanoComprimido);
  fseek (Salida, PosicionFinal, SEEK_SET);
  fclose (Entrada);

  /* Escribimos el ratio que se ha conseguido */
  for ( i=0 ; i<PUNTOINDICA+6 ; i++)
    { CursorAtras(); }
  Ratio = 100*(((float)TamanoComprimido/(float)TamanoOriginal))+0.5;
  printf ("%3d%%", Ratio);
  }

/*--------------------------------------------------------------------
 * FUNCION:  EscribeFalsaCabecera()
 * OBJETIVO: Escribir en el fichero de salida una serie de datos
 *           falsos que ms tarde sern sobreescritos con los correctos
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void EscribeFalsaCabecera (void)
  {
  char Aux[CABECERA];

  memset (Aux, 0, CABECERA);
  fwrite (Aux, 1, CABECERA, Salida);
  }

/*--------------------------------------------------------------------
 * FUNCION:  EscribeCabecera()
 * OBJETIVO: Escribir en el fichero de salida los datos de cabecera del
 *           fichero de entrada
 * ENTRADAS: El nombre del fichero, su tipo, mtodo de compresin,
 *           el tamao original y el comprimido
 * SALIDAS:  Ninguna
 * NOTA:     Los valores que ocupan 4 octetos se almacenan en orden
 *           MOTOROLA
 *------------------------------------------------------------------*/
void EscribeCabecera (char *Nombre, char Tipo, char Metodo,
                      unsigned long Original, unsigned long Comprimido)
  {
  char Aux[CABECERA];
  short int i;

  memset (Aux, 0, CABECERA);

  for ( i=0 ; Nombre[i] ; i++ )
    { Aux[i] = Nombre[i]; }

  i = LONG_NOMBRE;

  Aux[i++] = Tipo;
  Aux[i++] = Metodo;

  Aux[i++] = (char) (Original >> 24);
  Aux[i++] = (char) ((Original & 0x00FF0000) >> 16);
  Aux[i++] = (char) ((Original & 0x0000FF00) >> 8);
  Aux[i++] = (char) (Original & 0xFF);

  Aux[i++] = (char) (Comprimido >> 24);
  Aux[i++] = (char) ((Comprimido & 0x00FF0000) >> 16);
  Aux[i++] = (char) ((Comprimido & 0x0000FF00) >> 8);
  Aux[i++] = (char) (Comprimido & 0xFF);

  fwrite (Aux, 1, CABECERA, Salida);
  }

/*--------------------------------------------------------------------
 * FUNCION:  LeeOctetoPlano()
 * OBJETIVO: Leer un octeto del fichero de entrada
 * ENTRADAS: Una variable donde dejar el octeto
 * SALIDAS:  Lgica, que indica que se ha llegado al final del fichero
 *------------------------------------------------------------------*/
char LeeOctetoPlano (unsigned char *Dato)
  {
  int Lectura;
  char Respuesta;

  Lectura = getc (Entrada);
  if ( Lectura == EOF )
    { Respuesta = 0; }
  else
    {
    Respuesta = 1;
    *Dato = Lectura;
    ContadorParcial++;
    if ( ContadorParcial > Paso )
      {
      putchar ('o');
      ContadorParcial = 0;
      Punto++;
      }
    }

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  EscribeOctetoComprimido()
 * OBJETIVO: Escribir un octeto en el fichero de salida
 * ENTRADAS: El octeto
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void EscribeOctetoComprimido (unsigned char Dato)
  {
  putc (Dato, Salida);
  TamanoComprimido++;
  }

/*--------------------------------------------------------------------
 * FUNCION:  ComprimePlano()
 * OBJETIVO: Comprimir un fichero sin compresin (es decir, copiarlo)
 * ENTRADAS: Ninguna, se usan variables auxiliares
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void ComprimePlano (void)
  {
  unsigned char Dato;

  while ( LeeOctetoPlano (&Dato) )
    { EscribeOctetoComprimido (Dato); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ComprimeLZH()
 * OBJETIVO: Comprimir un fichero segn el algoritmo Lempel-Ziv usando
 *           codificacin Huffman
 * ENTRADAS: Ninguna, se usan variables auxiliares
 * SALIDAS:  Lgica, que indica si se ha podido comprimir
 *------------------------------------------------------------------*/
char ComprimeLZH (void)
  {
  int short i, r, s, Long, LongUltimaCoincidencia;
  unsigned char c;

  IniciaHuffman();
  IniciaArbol();
  s = 0;
  r = VENTANA - VER;
  for ( i=s ; i<r ; i++ )
    { Texto[i] = ' '; }
  for ( Long=0 ; Long<VER && LeeOctetoPlano(&c) ; Long++)
    { Texto [r+Long] = c; }

  for ( i=1 ; i<=VER ; i++ )
    { InsertaNodo (r-i); }
  InsertaNodo (r);

  do
    {
    if ( LongIgual > Long )
      { LongIgual = Long; }
    if ( LongIgual <= UMBRAL )
      {
      LongIgual = 1;
      CodificaOcteto (Texto[r]);
      }
    else
      {
      CodificaOcteto (255-UMBRAL+LongIgual);
      CodificaPosicion (PosIgual);
      }
    LongUltimaCoincidencia = LongIgual;
    for ( i=0 ; i<LongUltimaCoincidencia && LeeOctetoPlano(&c) ; i++ )
      {
      BorraNodo (s);
      Texto[s] = c;
      if ( s < VER-1 )
        { Texto[s+VENTANA] = c; }
      s = (s + 1) & (VENTANA - 1);
      r = (r + 1) & (VENTANA - 1);
      InsertaNodo (r);
      }
    while ( i++ < LongUltimaCoincidencia )
      {
      BorraNodo (s);
      s = (s + 1) & (VENTANA - 1);
      r = (r + 1) & (VENTANA - 1);
      if ( --Long )
        { InsertaNodo (r); }
      }
    } while ( Long > 0 );

  CodificaFinal();
  return ( TamanoComprimido < TamanoOriginal );
  }

/*--------------------------------------------------------------------
 * FUNCION:  IniciaArbol()
 * OBJETIVO: Dar valores iniciales al rbol de cadenas
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void IniciaArbol (void)
  {
  int short i;

  for ( i=VENTANA+1 ; i<=VENTANA+256 ; i++ )
    { Derecho[i] = HOJA; }    /* Raz */
  for ( i=0 ; i<VENTANA ; i++ )
    { Superior[i] = HOJA; }   /* Nodo */
  }

/*--------------------------------------------------------------------
 * FUNCION:  InsertaNodo()
 * OBJETIVO: Insertar un nodo en el rbol de cadenas
 * ENTRADAS: La posicin en Texto del comienzo de la cadena
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void InsertaNodo (int short r)
  {
  int short i, p, Comparacion;
  unsigned char *Clave;
  unsigned short c;

  Comparacion = 1;
  Clave = &Texto[r];
  p = VENTANA + 1 + Clave[0];
  Derecho[r] = Izquierdo[r] = HOJA;
  LongIgual = 0;
  for (;;)
    {
    if ( Comparacion>=0 )
      {
      if ( Derecho[p] != HOJA )
        { p = Derecho[p]; }
      else
        {
        Derecho[p] = r;
        Superior[r] = p;
        return;
        }
      }
    else
      {
      if ( Izquierdo[p] != HOJA )
        { p = Izquierdo[p]; }
      else
        {
        Izquierdo[p] = r;
        Superior[r] = p;
        return;
        }
      }

    for ( i=1 ; i<VER ; i++ )
      {
      if ( Comparacion = Clave[i]-Texto[p+i] )
        { break; }
      }

    if ( i>UMBRAL )
      {
      if ( i>LongIgual )
        {
        PosIgual = ( (r-p) & (VENTANA-1) ) - 1;
        if ( (LongIgual=i) >= VER )
          { break; }
        }
      if ( i==LongIgual )
        {
        if ( (c = ((r-p) & (VENTANA-1)) - 1) < PosIgual )
          { PosIgual = c; }
        }
      }
    }

  Superior[r] = Superior[p];
  Izquierdo[r] = Izquierdo[p];
  Derecho[r] = Derecho[p];
  Superior[Izquierdo[p]] = r;
  Superior[Derecho[p]] = r;
  if ( Derecho[Superior[p]] == p )
    { Derecho[Superior[p]] = r; }
  else
    { Izquierdo[Superior[p]] = r; }
  Superior[p] = HOJA;   /* Eliminamos p */
  }

/*--------------------------------------------------------------------
 * FUNCION:  BorraNodo()
 * OBJETIVO: Eliminar una cadena del rbol de cadenas
 * ENTRADAS: La posicin de la cadena
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void BorraNodo (int short p)
  {
  int short q;

  if ( Superior[p] == HOJA )
    { return; }

  if ( Derecho[p] == HOJA )
    { q = Izquierdo[p]; }
  else if (Izquierdo[p] == HOJA)
    { q = Derecho[p]; }
  else
    {
    q = Izquierdo[p];
    if (Derecho[q] != HOJA)
      {
      do
        {
        q = Derecho[q];
        } while ( Derecho[q] != HOJA );
      Derecho[Superior[q]] = Izquierdo[q];
      Superior[Izquierdo[q]] = Superior[q];
      Izquierdo[q] = Izquierdo[p];
      Superior[Izquierdo[p]] = q;
      }
    Derecho[q] = Derecho[p];
    Superior[Derecho[p]] = q;
    }

  Superior[q] = Superior[p];
  if ( Derecho[Superior[p]] == p )
    { Derecho[Superior[p]] = q; }
  else
    { Izquierdo[Superior[p]] = q; }

  Superior[p] = HOJA;
  }

/*--------------------------------------------------------------------
 * FUNCION:  EscribeBit()
 * OBJETIVO: Escribir varios bits
 * ENTRADAS: El nmero de bits y de dnde sacarlos
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void EscribeBit (int short l, unsigned short c)
  {
  AuxLZW |= c >> LongLZW;
  if ( (LongLZW += l) >= 8 )
    {
    EscribeOctetoComprimido (AuxLZW>>8);
    if ( (LongLZW -= 8) >= 8 )
      {
      EscribeOctetoComprimido (AuxLZW);
      LongLZW -= 8;
      AuxLZW = c << (l-LongLZW);
      }
    else
      { AuxLZW <<= 8; }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  CodificaOcteto()
 * OBJETIVO: Enviar a la salida un octeto codificndolo
 * ENTRADAS: El octeto
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void CodificaOcteto (unsigned short c)
  {
  unsigned short i;
  int short j, k;

  i = 0;
  j = 0;
  k = Padre [c+TABLA];

  /* Viajamos desde la hoja a la raz recogiendo bits por la izquierda */
  do
    {
    i >>= 1;
    if (k & 1)
      { i += 0x8000; }
    j++;
    } while ( (k = Padre[k]) != RAIZ );

  EscribeBit (j, i);
  Actualiza (c);
  }

/*--------------------------------------------------------------------
 * FUNCION:  CodificaPosicion()
 * OBJETIVO: Codificar una cadena
 * ENTRADAS: La posicin de la cadena
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void CodificaPosicion (unsigned short c)
  {
  unsigned short i;

  /* Sacamos los 6 bits superiores mirando en la tabla */
  i = c >> 6;
  EscribeBit (ValorLong[i], (unsigned) ValorCodigo[i] << 8);

  /* Sacamos los 6 bits inferiores tal cual */
  EscribeBit (6, (c & 0x3F) << 10);
  }

/*--------------------------------------------------------------------
 * FUNCION:  CodificaFinal()
 * OBJETIVO: Sacar los bits que faltan cuando se ha acabado
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void CodificaFinal (void)
  {
  if ( LongLZW )
    { EscribeOctetoComprimido (AuxLZW >> 8); }
  }