/*--------------------------------------------------------------------
 * FICHERO:  Solitari.c
 * OBJETIVO: Programa principal del Solitario
 * AUTOR:    Pedro Reina
 * FECHA:    L.9.3.1998
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * Ficheros de cabecera
 *------------------------------------------------------------------*/

#include <Olimpo.h>       /*  Versin 2.0  */

#include "Tablero.h"
#include "Movimien.h"
#include "Contador.h"
#include "Informac.h"

/*--------------------------------------------------------------------
 * Declaracin de funciones
 *------------------------------------------------------------------*/

void IniciaTodo();
void Explica();
void ExplicaObjetivo();
void ExplicaManejo();
void ExplicaAutor();
void ExplicaTeclas();
void Fichero();
void LeePartida();
void GrabaPartida();
void Opciones();
void Juego();
void EmpiezaJuego();
void PreparaPantallaJuego();
void SigueJuego();
void Juega();
void PartidaAtras();
void PartidaAdelante();
void Repite();
void MovimientoCero();
void MuevePosicion();
void PideLapso();
logico MuevePieza();
void Pausa();
cadena PideDirectorio();
cadena PideExtension();
cadena PideNombre();
cadena NombreCompleto();

/*--------------------------------------------------------------------
 * Variables globales
 *------------------------------------------------------------------*/

contador Fil=3, Col=3; /* Posicin del cursor */
contador Lapso = 2;    /* Espera entre dos repeticiones en automtico */

/*--------------------------------------------------------------------
 * Definicin de macros
 *------------------------------------------------------------------*/

#define LONG_DIRECTORIO  24
#define LONG_EXTENSION    3
#define LONG_NOMBRE       8

#define LimpiaZonaTrabajo()  Pan_Borra(NEGRO,4,0,21,79)
#define LimpiaZonaPequena()  Pan_Borra(NEGRO,10,0,21,79)

/*--------------------------------------------------------------------
 * Programa principal
 *------------------------------------------------------------------*/
void main()
  {
  enum { SALIDA_ESC, SALIDA, EXPLICA, JUEGO, FICHERO, OPCIONES };

  static cadena Principal[] = {">Salida", ">Explica", ">Juego", ">Fichero",
                               ">Opciones",  NIL };

  entero Opcion;
  logico Sigue;

  Pan_Define (PAN_TEXTO);
  IniciaTodo();
  Inf_Mejor = 0;

  Prg_Presenta ( "Solitario", "1.1", "Pedro Reina", "Marzo 1998" );

  Cdr_Caja (CDR_SIMPLE,1,0,3,79,NEGRO,VERDE);

  Opcion = JUEGO;
  Sigue = SI;
  while ( Sigue )
    {
    Opcion = Men_Horizontal (2,1,78,Principal,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:
      case SALIDA:
           if ( Usr_Consulta ("Quieres terminar el programa?") )
             { Sigue = NO; }
           Opcion = SALIDA;
           break;
      case EXPLICA:       Explica();       break;
      case JUEGO:         Juego();         break;
      case FICHERO:       Fichero();       break;
      case OPCIONES:      Opciones();      break;
      }
    LimpiaZonaTrabajo();
    }

  Pan_Cierra();
  }

/*--------------------------------------------------------------------
 * Definicin de funciones
 *------------------------------------------------------------------*/

/*--------------------------------------------------------------------
 * FUNCION:  Explica()
 * OBJETIVO: Dirigir las explicaciones
 *------------------------------------------------------------------*/
void Explica()
  {
  enum { SALIDA_ESC, OBJETIVO, MANEJO, AUTOR, TECLAS };
  static octeto MaxOpcion = 4;

  static cadena MenuExplica[] = { ">Objetivo del programa",
                                  ">Manejo de las opciones",
                                  ">Contactar con el autor",
                                  ">Teclas PC y teclas QL",
                                  NIL };

  static cadena Presentacion[] = {
    "Puedes pedir explicacin sobre diferentes aspectos",
    "de este programa. Elige el que desees y pulsa ESC",
    "cuando hayas terminado.",
    NIL };

  logico Sigue = SI;
  entero Opcion = 1, i;

  while ( Sigue )
    {
    LimpiaZonaTrabajo();
    Pan_Color (NEGRO,BLANCO);
    for ( i=0 ; Presentacion[i] ; i++ )
      { Pan_PonTexto (6+i,15,Presentacion[i]); }

    Opcion = Men_Vertical (11,29,14,52,MenuExplica,Opcion);
    LimpiaZonaTrabajo();
    switch ( Opcion )
      {
      case SALIDA_ESC:  Sigue = NO;         break;
      case OBJETIVO:    ExplicaObjetivo();  break;
      case MANEJO:      ExplicaManejo();    break;
      case AUTOR:       ExplicaAutor();     break;
      case TECLAS:      ExplicaTeclas();    break;
      }
    if ( Opcion )  { Usr_PulsaUnaTecla(""); }
    Opcion++;
    if ( Opcion > MaxOpcion )  { Opcion = 1; }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaObjetivo()
 * OBJETIVO: Explicar el objetivo del programa
 *------------------------------------------------------------------*/
void ExplicaObjetivo()
  {
  static cadena Mensaje[] = {
   "Este programa permite jugar al Solitario, conocido juego de mesa que",
   "se puede comprar en tiendas de juegos.",
   "",
   "Se parte de un tablero en forma de cruz lleno de piezas y el objetivo",
   "es dejar al final una sola.                                          ",
   "",
   "Lo primero es retirar una pieza, a eleccin del jugador. Despus hay ",
   "que ir capturando piezas. Las capturas se realizan en cualquier",
   "direccin, pasando la pieza que captura sobre la que se toma, cayendo",
   "sobre un hueco libre.",
   "",
   "Si el Solitario te parece sencillo y lo resuelves pronto, hay",
   "variaciones que lo hacen ms complicado. Por ejemplo: que la pieza",
   "que quede al final est en el hueco dejado por la primera pieza que",
   "quitaste, o fijar de antemano dnde debe quedar la ltima pieza, o...",
   NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (6+i,5,Mensaje[i]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaManejo()
 * OBJETIVO: Explicar cmo se maneja el programa
 *------------------------------------------------------------------*/
void ExplicaManejo()
  {
  static cadena Mensaje[] = {
    "La opcin Juego permite comenzar una partida nueva, seguir la que",
    "est en juego, ver repetida la partida y borrar la mejor puntuacin",
    "obtenida hasta el momento.",
    "",
    "La opcin Fichero permite grabar y leer partidas a ficheros.",
    "",
    "En Opciones se pueden definir dos cosas: si el sonido est encendido",
    "o apagado, y el lapso que debe pasar entre dos movimientos cuando",
    "se repiten partidas en modo automtico.",
    NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (5+i,4,Mensaje[i]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaAutor()
 * OBJETIVO: Explicar quin es el autor y cmo contactar con l
 *------------------------------------------------------------------*/
void ExplicaAutor()
  {
  static cadena Mensaje[] = {
    "Este programa es de Dominio Pblico. Lo puedes usar, distribuir",
    "y modificar como desees.",
    "",
    "Ha sido escrito en C usando el sistema de programacin Olimpo;",
    "si no dispones de la versin 2.0 de Olimpo, puedes cargarla",
    "directamente desde mi sede web o pedrmela si no dispones de",
    "acceso a Internet.",
    "",
    "Si tienes cualquier duda o consulta, cuntamela e intentar ayudarte.",
    "",
    "Mis datos:",
    "",
    "Pedro Reina",
    "c/ Marquesa de Argeso, 4      Telfono: 91 565 17 59",
    "28019 Madrid                   Correo electrnico: pedro@anit.es",
    "Espaa                         Web: www.anit.es/pedro",
    NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (5+i,5,Mensaje[i]); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  ExplicaTeclas()
 * OBJETIVO: Explicar las teclas que son distintas en QL y en PC
 *------------------------------------------------------------------*/
void ExplicaTeclas()
  {
  static cadena Mensaje[] = {
      "El QL y el PC tienen distintos teclados y algunas teclas",
      "no tienen correspondiente directo.",
      "",
      "Este programa asigna a ciertas combinaciones de teclas del QL",
      "la funcin de algunas teclas del PC. sta es la relacin:",
      "",
      "         Tecla PC        Equivalente QL",
      "         --------        --------------",
      "",
      "         Inicio          Alt-Izquierda",
      "         Fin             Alt-Derecha",
      "         RePag           Alt-Arriba",
      "         AvPag           Alt-Abajo",
      "         Insert          Maysculas-Fijamaysculas",
      "         Supr            Ctrl-Derecha",
      "         Retroceso       Ctrl-Izquierda",
       NIL };

  octeto i;

  Pan_Color (NEGRO,BLANCO);
  for ( i=0 ; Mensaje[i] ; i++ )
    { Pan_PonTexto (5+i,9,Mensaje[i]); }
  }
/*--------------------------------------------------------------------
 * FUNCION:  Fichero()
 * OBJETIVO: Presentar y manejar las opciones de manejo de ficheros
 * ENTRADAS: Ninguna
 * SALIDAS:  La partida se puede modificar
 *------------------------------------------------------------------*/
void Fichero()
  {
  enum { SALIDA_ESC, LEER, GRABAR, DIRECTORIO, EXTENSION };

  static cadena MenuFich[] = {">Leer", ">Grabar", ">Directorio",
                               ">Extensin", NIL };

  static char Directorio[LONG_DIRECTORIO+1] = "";
  static char Nombre[LONG_NOMBRE+1]         = "";
  static char Extension[LONG_EXTENSION+1]   = "slt";

  cadena   Aux;
  entero   Opcion;
  contador Fil=4, Col=20, Ancho=40, Alto=5;
  logico   Sigue = SI, Refresca = SI;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Opcion = LEER;
  while ( Sigue )
    {
    if ( Refresca )
      {
      Pan_Borra (NEGRO,Fil+3,Col+15,Fil+Alto-1,Col+Ancho-1);
      Pan_Color (NEGRO,VERDE);
      Pan_PonTexto (Fil+3,Col+15,Directorio);
      Pan_PonTexto (Fil+4,Col+15,Extension);
      Refresca = NO;
      }

    Opcion = Men_Vertical (Fil+1,Col+1,Fil+Alto-1,Col+14,MenuFich,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:  Sigue = NO;      break;
      case LEER:        Aux = PideNombre(Nombre);
                        if ( Tec_Ultima() != TEC_ESC )
                          {
                          Cad_Copia (Nombre,Aux);
                          LeePartida (Directorio,Nombre,Extension);
                          }
                        Cad_Destruye (Aux);
                        break;
      case GRABAR:
                        if ( Mov_Ultimo == MOV_NINGUNO )
                          {
                          Usr_Avisa ("La partida no tiene ningn movimiento");
                          }
                        else
                          {
                          Aux = PideNombre(Nombre);
                          if ( Tec_Ultima() != TEC_ESC )
                            {
                            Cad_Copia (Nombre,Aux);
                            GrabaPartida (Directorio,Nombre,Extension);
                            }
                          Cad_Destruye (Aux);
                          }
                        break;
      case DIRECTORIO:  Aux = PideDirectorio(Directorio);
                        if ( Tec_Ultima() != TEC_ESC )
                          {
                          Cad_Copia (Directorio,Aux);
                          Refresca = SI;
                          }
                        Cad_Destruye (Aux);
                        break;
      case EXTENSION:   Aux = PideExtension(Extension);
                        if ( Tec_Ultima() != TEC_ESC )
                          {
                          Cad_Copia (Extension,Aux);
                          Refresca = SI;
                          }
                        Cad_Destruye (Aux);
                        break;
      }
    LimpiaZonaPequena();
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  LeePartida()
 * OBJETIVO: Leer una partida
 * ENTRADAS: Directorio, nombre y extensin del fichero con la partida
 * SALIDAS:  La partida puede quedar modificada
 *------------------------------------------------------------------*/
void LeePartida (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  cadena   Completo;
  contador Tiempo;

  Completo = NombreCompleto(Dir,Nom,Ext);
  if ( (Tiempo = Mov_LeePartida (Completo)) != MOV_ERROR )
    {
    Cnt_Transcurrido = Tiempo;
    Mov_Actual = MOV_NINGUNO;
    Tab_Inicia();
    }
  Cad_Destruye (Completo);
  }

/*--------------------------------------------------------------------
 * FUNCION:  GrabaPartida()
 * OBJETIVO: Grabar una partida
 * ENTRADAS: Directorio, nombre y extensin del fichero donde dejar
 *           la partida
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void GrabaPartida (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  cadena Completo;

  Completo = NombreCompleto(Dir,Nom,Ext);
  Mov_GrabaPartida (Completo,Cnt_Transcurrido);

  Cad_Destruye (Completo);
  }

/*--------------------------------------------------------------------
 * FUNCION:  NombreCompleto()
 * OBJETIVO: Devolver el nombre completo de un fichero
 * ENTRADAS: Directorio, nombre y extensin del fichero
 * SALIDAS:  Una cadena con el nombre completo
 * NOTA:     La cadena hay que destruirla cuando no sea necesaria
 *------------------------------------------------------------------*/
cadena NombreCompleto (Dir,Nom,Ext)
cadena Dir,Nom,Ext;
  {
  cadena Respuesta, Aux;

  Aux = Fch_Nombre (Nom,Ext);
  Respuesta = Cad_Une (Dir,Aux,NIL);
  Cad_Destruye (Aux);

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideDirectorio()
 * OBJETIVO: Pedir al usuario un directorio
 * ENTRADAS: El directorio propuesto
 * SALIDAS:  El nuevo directorio
 *------------------------------------------------------------------*/
cadena PideDirectorio (Dir)
cadena Dir;
  {
  contador Fil=11, Col=22, Ancho=36, Alto=6;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+2,Col+6,"Escribe el directorio:");
  return ( Usr_Texto (Dir,LONG_DIRECTORIO,Fil+4,Col+6,NEGRO,BLANCO) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideExtension()
 * OBJETIVO: Pedir al usuario una extensin
 * ENTRADAS: La extensin propuesta
 * SALIDAS:  La nueva extensin
 *------------------------------------------------------------------*/
cadena PideExtension (Ext)
cadena Ext;
  {
  contador Fil=11, Col=22, Ancho=36, Alto=6;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+2,Col+6,"Escribe la extensin:");
  return ( Usr_Texto (Ext,LONG_EXTENSION,Fil+4,Col+6,NEGRO,BLANCO) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideNombre()
 * OBJETIVO: Pedir al usuario un nombre de fichero
 * ENTRADAS: El nombre propuesto
 * SALIDAS:  El nuevo nombre
 *------------------------------------------------------------------*/
cadena PideNombre (Nom)
cadena Nom;
  {
  contador Fil=11, Col=22, Ancho=36, Alto=6;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+2,Col+4,"Escribe el nombre del fichero:");
  return ( Usr_Texto (Nom,LONG_NOMBRE,Fil+4,Col+6,NEGRO,BLANCO) );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Opciones()
 * OBJETIVO: Seleccionar las opciones del programa
 * ENTRADAS: Ninguna
 * SALIDAS:  La variable global Lapso puede modificarse
 *------------------------------------------------------------------*/
void Opciones()
  {
  enum { SALIDA_ESC, SONIDO, LAPSO };

  static cadena MenuOpc[] = {">Sonido", ">Lapso", NIL };

  entero   Opcion;
  contador Fil=4, Col=66, Ancho=9, Alto=3;
  logico   Sigue = SI;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Opcion = SONIDO;
  while ( Sigue )
    {
    Opcion = Men_Vertical (Fil+1,Col+1,Fil+Alto-1,Col+Ancho-1,
                           MenuOpc,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:    Sigue = NO;      break;
      case SONIDO:
           if ( Usr_Consulta ("Quieres tener sonido en el programa?") )
             { Son_Enciende(); }
           else
             { Son_Apaga(); }
           break;
      case LAPSO:         PideLapso();     break;
      }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  PideLapso()
 * OBJETIVO: Fijar el lapso
 * ENTRADAS: Ninguna
 * SALIDAS:  La variable global Lapso puede modificarse
 *------------------------------------------------------------------*/
void PideLapso()
  {
  contador Fil=10, Col=25, Ancho=31, Alto=7;

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Pan_Color (NEGRO,VERDE);
  Pan_PonTexto (Fil+1,Col+2,"Fija el lapso entre dos");
  Pan_PonTexto (Fil+2,Col+2,"movimientos en la repeticin");
  Pan_PonTexto (Fil+3,Col+2,"automtica:");

  Lapso = Usr_Entero (Lapso,2,1,10,Fil+5,Col+14,NEGRO,BLANCO);
  Pan_Borra (NEGRO,Fil,Col,Fil+Alto,Col+Ancho);
  }

/*--------------------------------------------------------------------
 * FUNCION:  Juego()
 * OBJETIVO: Presentar las opciones para jugar
 * ENTRADAS: Ninguna
 * SALIDAS:  La partida puede quedar modificada
 *------------------------------------------------------------------*/
void Juego()
  {
  enum { SALIDA_ESC, SEGUIR, EMPEZAR, REPETIR, BORRARMEJOR };

  static cadena MenuJuego[] = {">Seguir", ">Empezar", ">Repetir",
                               ">Borrar mejor", NIL };

  entero   Opcion;
  contador Fil=4, Col=34, Ancho=15, Alto=5;
  logico   Sigue = SI;
  cadena   Preg =
           "Quieres poner a cero el contador de la mejor puntuacin de hoy?";

  Cdr_Caja (CDR_SIMPLE,Fil,Col,Fil+Alto,Col+Ancho,NEGRO,VERDE);

  Opcion = SEGUIR;
  while ( Sigue )
    {
    Opcion = Men_Vertical (Fil+1,Col+1,Fil+Alto-1,Col+Ancho-1,
                           MenuJuego,Opcion);
    switch ( Opcion )
      {
      case SALIDA_ESC:   Sigue = NO;                   break;
      case SEGUIR:       SigueJuego();   Sigue = NO;   break;
      case EMPEZAR:      EmpiezaJuego(); Sigue = NO;   break;
      case REPETIR:
                         if ( Mov_Ultimo == MOV_NINGUNO )
                           {
                           Usr_Avisa ("La partida no tiene ningn movimiento");
                           }
                         else
                           {
                           Repite();
                           Sigue = NO;
                           }
                         break;
      case BORRARMEJOR:  if ( Usr_Consulta (Preg) )
                           {
                           Inf_Mejor = 0;
                           Usr_PulsaUnaTecla ("Mejor de hoy puesto a cero.");
                           }
                         break;
      }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  EmpiezaJuego()
 * OBJETIVO: Empezar una nueva partida
 * ENTRADAS: Ninguna
 * SALIDAS:  La partida queda modificada
 *------------------------------------------------------------------*/
void EmpiezaJuego()
  {
  IniciaTodo();
  SigueJuego();
  }

/*--------------------------------------------------------------------
 * FUNCION:  IniciaTodo()
 * OBJETIVO: Iniciar todos los objetos necesarios
 * ENTRADAS: Ninguna
 * SALIDAS:  La partida queda modificada
 *------------------------------------------------------------------*/
void IniciaTodo()
  {
  Tab_Inicia();
  Mov_Inicia();
  Cnt_Inicia();
  Inf_Inicia();
  }

/*--------------------------------------------------------------------
 * FUNCION:  PreparaPantallaJuego()
 * OBJETIVO: Dejar la pantalla preparada para el juego
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void PreparaPantallaJuego()
  {
  LimpiaZonaTrabajo();
  Mov_DibujaMarco();
  Mov_ActualizaPantalla();
  Cnt_DibujaMarco();
  Cnt_Escribe();
  Inf_DibujaMarco();
  Inf_ActualizaPantalla();
  Tab_DibujaTodo();
  }

/*--------------------------------------------------------------------
 * FUNCION:  SigueJuego()
 * OBJETIVO: Continuar la partida donde se hubiera quedado
 * ENTRADAS: Ninguna
 * SALIDAS:  La partida puede quedar modificada
 *------------------------------------------------------------------*/
void SigueJuego()
  {
  PreparaPantallaJuego();
  Juega();
  }

/*--------------------------------------------------------------------
 * FUNCION:  Juega()
 * OBJETIVO: Jugar, realizando movimientos y teniendo opciones
 * ENTRADAS: Ninguna
 * SALIDAS:  La partida puede quedar modificada
 *------------------------------------------------------------------*/
void Juega()
  {
  static tecla ListaTec[] = { TEC_IZQUIERDA, TEC_DERECHA, TEC_ARRIBA,
                              TEC_ABAJO, TEC_ESPACIO, TEC_ESC,
                              TEC_A, TEC_MAY_A, TEC_S, TEC_MAY_S,
                              TEC_P, TEC_MAY_P, NIL };
  cadena   Mens1, Mens2;
  contador Movimiento[4];
  contador F2, C2;
  tecla    Tecla;
  logico   Sigue = SI, Refresca = SI;

  Mens1 =
  "CURSOR: elige casilla.  ESPACIO: acepta.  P: pausa.  ESC: men principal";
  Mens2 =
  "A: anterior movimiento.  S: siguiente movimiento";

  Tab_Marca (Fil,Col);
  Cnt_Enciende();

  while ( Sigue )
    {
    if ( Mov_Actual == MOV_NINGUNO )
      {
      MovimientoCero (&Fil,&Col);
      Sigue = Tec_Ultima() != TEC_ESC;
      Refresca = SI;
      }
    else
      {
      if ( !Tab_HayPosibilidad() )
        {
        if ( Inf_Retiradas == INF_PIEZAS - 1 )
          { Usr_PulsaUnaTecla (
               "Enhorabuena: has terminado con xito el solitario.");
          }
        else
          { Usr_PulsaUnaTecla (
               "Lo siento, no te quedan movimientos. Has perdido.");
          }
        Sigue = NO;
        }
      else
        {
        if ( Refresca )
          {
          Usr_Indica (Mens1,Mens2);
          Pan_CursorVisible (NO);
          Refresca = NO;
          }
        if ( Tecla = Tec_ValidadaRapido (ListaTec) )
          {
          switch (Tecla)
            {
            case TEC_IZQUIERDA:
            case TEC_DERECHA:
            case TEC_ARRIBA:
            case TEC_ABAJO:
                               MuevePosicion (&Fil,&Col,Tecla);  break;
            case TEC_ESPACIO:
                               if ( MuevePieza (Fil,Col,&F2,&C2) )
                                 {
                                 Movimiento[0] = Fil;
                                 Movimiento[1] = Col;
                                 Movimiento[2] = F2;
                                 Movimiento[3] = C2;
                                 Tab_Mueve (Movimiento);
                                 Mov_Almacena (Movimiento);
                                 Inf_Incrementa();
                                 Fil=F2, Col=C2;
                                 Tab_Marca (Fil,Col);
                                 }
                               Refresca = SI;
                               break;
            case TEC_ESC:      Sigue = NO;        break;
            case TEC_P:
            case TEC_MAY_P:    Pausa();
                               Refresca = SI;
                               Tab_Marca (Fil,Col);
                               break;
            case TEC_A:
            case TEC_MAY_A:    PartidaAtras(1);   break;
            case TEC_S:
            case TEC_MAY_S:    PartidaAdelante(); break;
            } /* Fin switch Tecla */
          }
        else
          {
          Cnt_Aumenta();
          } /* Fin if Tecla */
        } /* Fin if !Tab_HayPosibilidad() */
      } /* Fin if Mov_Actual==MOV_NINGUNO */
    } /* Fin while Sigue */
  }

/*--------------------------------------------------------------------
 * FUNCION:  PartidaAtras()
 * OBJETIVO: Llevar toda la partida un movimiento hacia atrs
 * ENTRADAS: El nmero de movimiento mnimo que se admite ejecutar
 * SALIDAS:  La posicin puede quedar modificada
 *------------------------------------------------------------------*/
void PartidaAtras (Num)
contador Num;
  {
  contador Movimiento[4];

  if ( Mov_Anterior(Movimiento,Num) )
    {
    Tab_Dibuja (Fil,Col);
    Tab_MueveAtras (Movimiento);
    Mov_ActualizaPantalla();
    Fil = Movimiento[0];
    Col = Movimiento[1];
    if ( Fil==0 && Col==0 )
      {
      Fil = Movimiento[2];
      Col = Movimiento[3];
      }
    Tab_Marca (Fil,Col);
    Inf_Decrementa();
    }
  else { Son_MalaTecla(); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  PartidaAdelante()
 * OBJETIVO: Llevar toda la partida un movimiento hacia delante
 * ENTRADAS: Ninguna
 * SALIDAS:  La posicin puede quedar modificada
 *------------------------------------------------------------------*/
void PartidaAdelante ()
  {
  contador Movimiento[4];

  if ( Mov_Siguiente(Movimiento) )
    {
    Tab_Dibuja (Fil,Col);
    Tab_Mueve (Movimiento);
    Mov_ActualizaPantalla();
    Fil = Movimiento[2];
    Col = Movimiento[3];
    if ( Fil==0 && Col==0 )
      {
      Fil = Movimiento[0];
      Col = Movimiento[1];
      }
    Tab_Marca (Fil,Col);
    Inf_Incrementa();
    }
  else { Son_MalaTecla(); }
  }

/*--------------------------------------------------------------------
 * FUNCION:  MovimientoCero()
 * OBJETIVO: Realizar el movimiento cero de la partida: quitar una pieza
 * ENTRADAS: Las variables fila y columna donde est el cursor
 * SALIDAS:  La partida puede quedar modificada
 *------------------------------------------------------------------*/
void MovimientoCero (Fil,Col)
contador *Fil, *Col;
  {
  static tecla ListaTec[] = { TEC_IZQUIERDA, TEC_DERECHA, TEC_ARRIBA,
                              TEC_ABAJO, TEC_ENTER, TEC_ESC, NIL };
  tecla    Tecla;
  logico   Sigue = SI;
  contador Movimiento[4];

  Usr_Indica (
    "Movimiento cero: hay que retirar una pieza. ESC: anula",
    "CURSOR: elige la pieza que vas a quitar.  ENTER: la quita.");

  while ( Sigue )
    {
    if ( Tecla = Tec_ValidadaRapido (ListaTec) )
      {
      switch ( Tecla )
        {
        case TEC_IZQUIERDA:
        case TEC_DERECHA:
        case TEC_ARRIBA:
        case TEC_ABAJO:
                           MuevePosicion (Fil,Col,Tecla);    break;
        case TEC_ESC:      Sigue = NO;                       break;
        case TEC_ENTER:    Tab_PonCelda (*Fil,*Col,TAB_HUECO);
                           Tab_Dibuja (*Fil,*Col);
                           Tab_Marca (*Fil,*Col);

                           Movimiento[0] = *Fil;
                           Movimiento[1] = *Col;
                           Movimiento[2] = 0;
                           Movimiento[3] = 0;
                           Mov_Almacena (Movimiento);
                           Inf_Incrementa();

                           Sigue = NO;
                           break;
        }
      }
    else { Cnt_Aumenta(); }
    }
  }

/*--------------------------------------------------------------------
 * FUNCION:  Repite()
 * OBJETIVO: Repetir el desarrollo de la partida, sin modificarla
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Repite()
  {
  enum { AUTO, MANUAL };                 /* Posibles modos             */
  enum { PAUSA  ,    MEDIA, FINAL };     /* Situaciones en modo AUTO   */
  enum { INICIAL, /* MEDIA, FINAL */ };  /* Situaciones en modo MANUAL */
  /* Precaucin: PAUSA e INICIAL valen 0, se distinguen en el modo */

  static cadena PrimeraLinea[2] = {
         "Repeticin automtica de partida.  M: paso a manual.  ESC: salir",
         "Repeticin manual de partida.  T: paso a automtica.  ESC: salir" };

  static cadena SegundaLinea[2][3] = {
         "Situacin de pausa.  P: seguir",
         "P: pausa en la repeticin",
         "Posicin final",
         "Posicin inicial.  S: siguiente movimiento",
         "A: anterior movimiento.  S: siguiente movimiento",
         "Posicin final.  A: anterior movimiento" };

  static tecla ListaTec[2][3][8] = {
         TEC_ESC, TEC_M, TEC_MAY_M, TEC_P, TEC_MAY_P, NIL,   NIL,       NIL,
         TEC_ESC, TEC_M, TEC_MAY_M, TEC_P, TEC_MAY_P, NIL,   NIL,       NIL,
         TEC_ESC, TEC_M, TEC_MAY_M, NIL,   NIL,       NIL,   NIL,       NIL,
         TEC_ESC, TEC_T, TEC_MAY_T, TEC_S, TEC_MAY_S, NIL,   NIL,       NIL,
         TEC_ESC, TEC_T, TEC_MAY_T, TEC_A, TEC_MAY_A, TEC_S, TEC_MAY_S, NIL,
         TEC_ESC, TEC_T, TEC_MAY_T, TEC_A, TEC_MAY_A, NIL,   NIL,       NIL };

  logico   Sigue = SI, Refresca = SI;
  contador Modo = AUTO, Situacion = MEDIA;
  tecla    Tecla;

  Mov_Actual = MOV_NINGUNO;
  Inf_Inicia();
  Tab_Inicia();
  PreparaPantallaJuego();

  while ( Sigue )
    {
    if ( ! (Modo == AUTO && Situacion == PAUSA) )
      {
      if ( Mov_Actual == MOV_NINGUNO )
        {
        if ( Modo == AUTO ) { Situacion = MEDIA; }
        else                { Situacion = INICIAL; }
        }
      else if ( Mov_Actual == Mov_Ultimo  ) { Situacion = FINAL; }
      else                                  { Situacion = MEDIA; }
      }

    if ( Refresca )
      {
      Usr_Indica (PrimeraLinea[Modo], SegundaLinea[Modo][Situacion]);
      Refresca = NO;
      }

    if ( Tecla = Tec_ValidadaRapido (ListaTec[Modo][Situacion]) )
      {
      switch ( Tecla )
        {
        case TEC_ESC:    Sigue = NO;                   break;
        case TEC_M:
        case TEC_MAY_M:
        case TEC_T:
        case TEC_MAY_T:  Modo = ! Modo; Refresca = SI; break;
        case TEC_P:
        case TEC_MAY_P:
                         if ( Modo == AUTO && Situacion == PAUSA )
                           { Situacion = MEDIA; }
                         else { Situacion = PAUSA; }
                         Refresca = SI;
                         break;
        case TEC_A:
        case TEC_MAY_A:  PartidaAtras(0);   Refresca = SI; break;
        case TEC_S:
        case TEC_MAY_S:  PartidaAdelante(); Refresca = SI; break;
        } /* Fin switch Tecla */
      }
    else
      {
      if ( Modo == AUTO && Situacion != PAUSA )
        {
        Tim_Espera (Lapso);
        PartidaAdelante();
        Refresca = SI;
        }
      } /* Fin if Tecla */
    } /* Fin while Sigue */
  }

/*--------------------------------------------------------------------
 * FUNCION:  MuevePosicion()
 * OBJETIVO: Intentar mover el cursor en la direccin indicada
 * ENTRADAS: Las variables fila y columna donde est el cursor y la tecla
 *           que indica la direccin del movimiento
 * SALIDAS:  Las variables fila y columna pueden quedar modificadas
 *------------------------------------------------------------------*/
void MuevePosicion (Fil,Col,Tecla)
contador *Fil, *Col;
tecla     Tecla;
  {
  contador Hor, Ver, NuevaF, NuevaC;

  Tab_Dibuja (*Fil,*Col);

  switch ( Tecla )
    {
    case TEC_IZQUIERDA:  Hor = -1 , Ver =  0;   break;
    case TEC_DERECHA:    Hor =  1 , Ver =  0;   break;
    case TEC_ARRIBA:     Hor =  0 , Ver = -1;   break;
    case TEC_ABAJO:      Hor =  0 , Ver =  1;   break;
    }

  NuevaF = *Fil + Ver;
  NuevaC = *Col + Hor;

  if ( Tab_PosVale (NuevaF,NuevaC) )
    {
    *Fil = NuevaF;
    *Col = NuevaC;
    }

  Tab_Marca (*Fil,*Col);
  }

/*--------------------------------------------------------------------
 * FUNCION:  MuevePieza()
 * OBJETIVO: Realizar una captura con una pieza, si es posible
 * ENTRADAS: Las fila y columna donde est el cursor y las variables
 *           donde se anotar la posicin final
 * SALIDAS:  Lgica, indicando si se ha realizado captura
 *           La partida puede quedar modificada, las variables de la
 *           posicin final quedan modificadas
 *------------------------------------------------------------------*/
logico MuevePieza (Fil,Col,F2,C2)
contador Fil, Col, *F2, *C2;
  {
  static contador F[] = {-2,-2,-2,0,2,2,2,0};
  static contador C[] = {-2,0,2,2,2,0,-2,-2};

  static tecla ListaTec[] = { TEC_ESPACIO, TEC_ENTER, TEC_ESC, NIL };

  contador i = 0;
  logico   Respuesta, Decidido = NO;
  cadena   Mens1, Mens2;
  tecla    Tecla;

  Mens1 = "Elige el hueco donde vas a poner la pieza.  ESC: anula.";
  Mens2 = "ESPACIO: va indicando todas las posibilidades.  ENTER: acepta.";

  if ( Tab_Celda(Fil,Col) == TAB_HUECO )
    {
    Son_MalaTecla();
    Usr_PulsaUnaTecla ("En esa casilla no hay pieza.");
    Respuesta = NO;
    }

  else if ( !Tab_AlgunaCaptura (Fil,Col) )
    {
    Son_MalaTecla();
    Usr_PulsaUnaTecla ("Esa pieza no puede realizar ninguna captura.");
    Respuesta = NO;
    }

  else
    {
    Usr_Indica (Mens1,Mens2);
    do
      {
      while ( !Tab_PuedeCapturar (Fil,Col,Fil+F[i],Col+C[i]) )
        { i = (i==7) ? 0 : i+1; }
      Tab_Marca (Fil+F[i],Col+C[i]);
      if ( Tecla = Tec_ValidadaRapido (ListaTec) )
        {
        switch ( Tecla )
          {
          case TEC_ESPACIO: Tab_Dibuja (Fil+F[i],Col+C[i]);
                            i = (i==7) ? 0 : i+1;
                            break;
          case TEC_ESC:     Respuesta = NO, Decidido = SI;
                            Tab_Dibuja (Fil+F[i],Col+C[i]);
                            break;
          case TEC_ENTER:   Respuesta = SI, Decidido = SI;
                            *F2 = Fil + F[i];
                            *C2 = Col + C[i];
                            break;
          }
        }
      else
        {
        Cnt_Aumenta();
        } /* Fin if Tecla validada */
      } while ( !Decidido );
    } /* Fin cadena de if */

  return ( Respuesta );
  }

/*--------------------------------------------------------------------
 * FUNCION:  Pausa()
 * OBJETIVO: Parar el juego hasta que quiera el jugador
 * ENTRADAS: Ninguna
 * SALIDAS:  Ninguna
 *------------------------------------------------------------------*/
void Pausa()
  {
  Tab_Disimula();
  Usr_PulsaUnaTecla ("El juego est en situacin de PAUSA");
  Tab_DibujaPiezas();
  Cnt_Enciende();
  }