#!/usr/bin/perl -w

#--------------------------------------------------------------------
# Fichero:  uso.pl
# Objetivo: Actualizar las bases de datos y grficos de uso
# Autor:    Pedro Reina <pedro@pedroreina.net>
# Fecha:    S.10.11.2007
# Licencia: Dominio pblico
#           http://creativecommons.org/licenses/publicdomain/
#--------------------------------------------------------------------

#---------------------------------
# Declaracin de mdulos
#---------------------------------

use strict;                 # Comprobaciones estrictas

#---------------------------------
# Declaracin de constantes
#---------------------------------

# El directorio donde estn todos los archivos
my $Dir = "/home/web/uso/";

# Las variables y sus nombres humamos
my %Nombre =
  (
  "carga",      "Carga del sistema",
  "memoria",    "Memoria usada",
  "procesos",   "Procesos",
  "conexiones", "Conexiones de red",
  "usuarios",   "Usuarios activos",
  "trafico",    "Trfico de red"
  );

# La orden para obtener la carga del sistema los ltimos cinco minutos
my $OrdenCarga = "cat /proc/loadavg | cut -d ' ' -f 2";

# La orden para obtener la cantidad de memoria realmente usada
# (Obsrvese que el smbolo "$" va escapado)
my $OrdenMemoria = "cat /proc/meminfo | grep Committed_AS | " .
                   "awk '{ print \$2 }'";

# La orden para obtener el nmero de procesos en ejecucin
my $OrdenProcesos = "cat /proc/loadavg | cut -d ' ' -f 4 | cut -d '/' -f 2";

# La orden para obtener el nmero de conexiones TCP activas
my $OrdenConexiones = "netstat -tn | grep ESTABLISHED | wc -l";

# La orden para obtener el nmero de usuarios (shells) activos
my $OrdenUsuarios = "w -h | wc -l";

# La orden para obtener la cantidad de bytes que entran por la red
# (Obsrvese que el smbolo "$" va escapado)
my $OrdenTraficoEntrante = "cat /proc/net/dev | grep eth0 | " .
                           "awk '{ print \$1 }' | cut -d ':' -f 2";

# La orden para obtener la cantidad de bytes que salen por la red
# (Obsrvese que el smbolo "$" va escapado)
my $OrdenTraficoSaliente = "cat /proc/net/dev | grep eth0 | " .
                           "awk '{ print \$9 }'";

# La orden para ejecutar RRDtool
my $RRDtool = "/usr/bin/rrdtool";

# La orden para actualizar la base de datos SQLite
my $SQLite = "/usr/bin/sqlite3 " . $Dir . "uso.db";

# Las dimensiones de los grficos generados
my $Anchura = 600;
my $Altura = 200;

#---------------------------------
# Declaracin de variables
#---------------------------------

my $Carga;
my $Memoria;
my $Procesos;
my $Conexiones;
my $Usuarios;
my $TraficoEntrante;
my $TraficoSaliente;

#---------------------------------
# Programa principal
#---------------------------------

# Para ayudar al desarrollo y depuracin de este programa,
# cada paso individual es una funcin diferente y en cada
# una de ellas se puede activar independientemente la depuracin
# y la ejecucin

# Leemos los valores de las variables
LeerValores();

# Actualizamos las bases de datos RRD
ActualizaBaseDatoRRD();

# Actualizamos la base de datos SQLite
ActualizaBaseDatoSQLite();

# Generamos los grficos diarios
GeneraGraficoDiario();

# Generamos los grficos semanales
GeneraGraficoSemanal();

# Generamos los grficos mensuales
GeneraGraficoMensual();

# Generamos los grficos anuales
GeneraGraficoAnual();

#---------------------------------
# Funciones
#---------------------------------

#--------------------------------------------------------------------
# Funcin:  LeerValores
# Objetivo: Leer los valores de las variables
#--------------------------------------------------------------------

sub LeerValores
  {
  my $Depurar = 0;

  $Carga = EjecutaOrden ($OrdenCarga);
  $Memoria = EjecutaOrden ($OrdenMemoria);
  $Procesos = EjecutaOrden ($OrdenProcesos);
  $Conexiones = EjecutaOrden ($OrdenConexiones);
  $Usuarios = EjecutaOrden ($OrdenUsuarios);
  $TraficoEntrante = EjecutaOrden ($OrdenTraficoEntrante);
  $TraficoSaliente = EjecutaOrden ($OrdenTraficoSaliente);

  if ( $Depurar )
    {
    print "Carga: $Carga\n";
    print "Memoria: $Memoria\n";
    print "Procesos: $Procesos\n";
    print "Conexiones: $Conexiones\n";
    print "Usuarios: $Usuarios\n";
    print "TraficoEntrante: $TraficoEntrante\n";
    print "TraficoSaliente: $TraficoSaliente\n";
    }
  }

#--------------------------------------------------------------------
# Funcin:  ActualizaBaseDatoRRD
# Objetivo: Actualizar las bases de datos RRD con nuevos valores
#--------------------------------------------------------------------

sub ActualizaBaseDatoRRD
  {
  ActualizaRRD ("carga", $Carga);
  ActualizaRRD ("memoria", $Memoria);
  ActualizaRRD ("procesos", $Procesos);
  ActualizaRRD ("conexiones", $Conexiones);
  ActualizaRRD ("usuarios", $Usuarios);
  ActualizaRRD ("trafico", "$TraficoEntrante:$TraficoSaliente");
  }

#--------------------------------------------------------------------
# Funcin:  ActualizaRRD
# Objetivo: Actualizar una base de datos RRD con un valor
#--------------------------------------------------------------------

sub ActualizaRRD
  {
  my $Depurar = 0;
  my $Ejecutar = 1;

  # La base de datos es el primer parmetro
  my $BaseDatos = $_[0];

  # El valor es el segundo parmetro
  my $Valor = $_[1];

  # La orden para hacer la actualizacin
  my $Orden = "$RRDtool update $Dir$BaseDatos.rrd N:$Valor";

  if ( $Depurar )
    { print "$Orden\n"; }

  if ( $Ejecutar )
    { qx ($Orden); }
  }

#--------------------------------------------------------------------
# Funcin:  ActualizaBaseDatoSQLite
# Objetivo: Actualizar la base de datos SQLite
#--------------------------------------------------------------------

sub ActualizaBaseDatoSQLite
  {
  my $Depurar = 0;
  my $Ejecutar = 1;

  my $Consulta;
  my $Orden;

  $Consulta = "\"INSERT INTO uso " .
  "(carga, memoria, procesos, conexiones, usuarios, entrante, saliente) " .
  "VALUES ($Carga, $Memoria, $Procesos, $Conexiones, $Usuarios, " .
           "$TraficoEntrante, $TraficoSaliente)\"";

  $Orden = $SQLite . " " . $Consulta;

  if ( $Depurar )
    { print "$Orden\n"; }

  if ( $Ejecutar )
    { qx ($Orden); }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoDiario
# Objetivo: Generar los grficos diarios de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoDiario
  {
  my $Variable;

  while ( $Variable = each (%Nombre) )
    { GeneraGrafico ($Variable, "diario"); }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoSemanal
# Objetivo: Generar los grficos semanales de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoSemanal
  {
  my $Variable;

  # Averiguamos la hora local
  my ($seg, $min) = localtime(time);

  # Los datos para obtener los grficos semanales se actualizan
  # cada treinta minutos, as que generamos los grficos a ese ritmo
  if ( $min % 30 == 0 )
    {
    while ( $Variable = each (%Nombre) )
      { GeneraGrafico ($Variable, "semanal"); }
    }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoMensual
# Objetivo: Generar los grficos mensuales de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoMensual
  {
  my $Variable;

  # Averiguamos la hora local
  my ($seg, $min, $hora) = localtime(time);

  # Los datos para obtener los grficos mensuales se actualizan
  # cada dos horas, as que generamos los grficos a ese ritmo
  if ( ($min == 0) && ($hora % 2 == 0) )
    {
    while ( $Variable = each (%Nombre) )
      { GeneraGrafico ($Variable, "mensual"); }
    }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGraficoAnual
# Objetivo: Generar los grficos anuales de todas las variables
#--------------------------------------------------------------------

sub GeneraGraficoAnual
  {
  my $Variable;

  # Averiguamos la hora local
  my ($seg, $min, $hora) = localtime(time);

  # Los datos para obtener los grficos mensuales se actualizan
  # cada da, as que generamos los grficos a ese ritmo
  if ( ($hora == 0) && ($min == 0) )
    {
    while ( $Variable = each (%Nombre) )
      { GeneraGrafico ($Variable, "anual"); }
    }
  }

#--------------------------------------------------------------------
# Funcin:  GeneraGrafico
# Objetivo: Generar un grfico de un tipo de una variable
#--------------------------------------------------------------------

sub GeneraGrafico
  {
  my $Depurar = 0;
  my $Ejecutar = 1;
  my $Orden;
  my $Extension;
  my $EjeHorizontal;

  # La variable es el primer parmetro
  my $Variable = $_[0];

  # El tipo de grfico es el segundo parmetro
  my $Tipo = $_[1];

  # El archivo final del grfico
  my $Archivo = "$Dir$Variable-$Tipo.png";

  # El ttulo del grfico
  my $Titulo = $Nombre{"$Variable"};

  # La extensin en el tiempo y el eje horizontal dependen del tipo
  if ( $Tipo eq "diario" )
    {
    $Extension = "end-28h ";
    $EjeHorizontal = "MINUTE:15:HOUR:1:HOUR:1:0:%H ";
    }
  elsif ( $Tipo eq "semanal" )
    {
    $Extension = "end-9d ";
    $EjeHorizontal = "HOUR:6:DAY:1:DAY:1:86400:%a%t%d ";
    }
  elsif ( $Tipo eq "mensual" )
    {
    $Extension = "end-5w ";
    $EjeHorizontal = "DAY:1:WEEK:1:WEEK:1:604800:Semana%t%V ";
    }
  elsif ( $Tipo eq "anual" )
    {
    $Extension = "end-13mon ";
    $EjeHorizontal = "WEEK:1:MONTH:1:MONTH:1:2592000:%b ";
    }

  # Preparamos la orden para generar el grfico en tres fases

  # Primera fase de la orden, los parmetros ms generales
  $Orden = "$RRDtool graph $Archivo " .
           "--tabwidth 6 " .
           "--title '$Titulo' " .
           "--start $Extension " .
           "--width $Anchura --height $Altura " .
           "--x-grid $EjeHorizontal ";

  # Segunda fase de la orden, parmetros particulares
  if ( $Variable eq "carga" )
    { $Orden .= "--units-exponent 0 "; }
  if ( $Variable eq "usuarios" )
    { $Orden .= "--y-grid 1:1 "; }

  # Tercera fase de la orden, parmetros especficos
  # El grfico diario solo tiene los datos puros
  if ( $Tipo eq "diario" )
    {
    if ( $Variable eq "memoria" )
      {
      $Orden .= "DEF:$Variable=$Dir$Variable.rrd:$Variable:AVERAGE " .
                "CDEF:megas=$Variable,1024,* " .
                "AREA:megas#5d3fb5";
      }

    elsif ( $Variable ne "trafico" )
      {
      $Orden .= "DEF:$Variable=$Dir$Variable.rrd:$Variable:AVERAGE " .
                "AREA:$Variable#5d3fb5";
      }

    else
      {
      $Orden .= "DEF:entrante=$Dir$Variable.rrd:entrante:AVERAGE " .
                "DEF:saliente=$Dir$Variable.rrd:saliente:AVERAGE " .
                "AREA:saliente#5d3fb5:Saliente " .
                "LINE1:entrante#FF0000:Entrante";
      }
    }

  # Los dems grficos tienen mximo, media y mnimo
  else
    {
    if ( $Variable eq "memoria" )
      {
      $Orden .= "DEF:$Variable-Med=$Dir$Variable.rrd:$Variable:AVERAGE " .
                "DEF:$Variable-Min=$Dir$Variable.rrd:$Variable:MIN " .
                "DEF:$Variable-Max=$Dir$Variable.rrd:$Variable:MAX " .
                "CDEF:megasMed=$Variable-Med,1024,* " .
                "CDEF:megasMin=$Variable-Min,1024,* " .
                "CDEF:megasMax=$Variable-Max,1024,* " .
                "AREA:megasMax#FF0000:Mximo " .
                "LINE2:megasMed#5d3fb5:Media "  .
                "AREA:megasMin#00FF00:Mnimo ";
      }

    elsif ( $Variable ne "trafico" )
      {
      $Orden .= "DEF:$Variable-Med=$Dir$Variable.rrd:$Variable:AVERAGE " .
                "DEF:$Variable-Min=$Dir$Variable.rrd:$Variable:MIN " .
                "DEF:$Variable-Max=$Dir$Variable.rrd:$Variable:MAX " .
                "AREA:$Variable-Max#FF0000:Mximo " .
                "LINE2:$Variable-Med#5d3fb5:Media " .
                "AREA:$Variable-Min#00FF00:Mnimo";
      }

    else
      {
      $Orden .= "DEF:saliente-Med=$Dir$Variable.rrd:saliente:AVERAGE " .
                "DEF:saliente-Min=$Dir$Variable.rrd:saliente:MIN " .
                "DEF:saliente-Max=$Dir$Variable.rrd:saliente:MAX " .
                "AREA:saliente-Max#FF0000:Mximo " .
                "LINE2:saliente-Med#5d3fb5:Media " .
                "AREA:saliente-Min#00FF00:Mnimo";
      }
    }

  if ( $Depurar )
    { print "$Orden\n"; }

  if ( $Ejecutar )
    { qx ($Orden); }
  }

#--------------------------------------------------------------------
# Funcin:  EjecutaOrden
# Objetivo: Ejecutar una orden del sistema y devolver el resultado
#--------------------------------------------------------------------

sub EjecutaOrden
  {
  # La orden es el primer parmetro
  my $Orden = $_[0];

  # Ejecutamos la orden
  my $Resultado = qx ($Orden);

  # Quitamos el fin de lnea
  chop ($Resultado);

  # Devolvemos el resultado
  return $Resultado;
  }
