<?php

//--------------------------------------------------------------
// Fichero:  basedato.php
// Objetivo: Manejo de bases de datos
// Autor:    Pedro Reina <pedro@pedroreina.net>
// Fecha:    D.31.1.2016
// Licencia: Dominio público
//--------------------------------------------------------------

//------------------------------------------------
// Ficheros necesarios
//------------------------------------------------

//----------------------------
// Constantes
//----------------------------

//----------------------------
// Variables globales
//----------------------------

//----------------------------
// Funciones que ofrecemos
//----------------------------

// Devuelve un único valor consultado a la base de datos
function TedeceaBD_ConsultaValor ($Archivo, $Consulta)
  {
  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Ejecutamos la consulta
  $Resultado = $Conexion->query ($Consulta);

  // Nos quedamos el resultado
  $Respuesta = $Resultado->fetchColumn();

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el resultado
  return $Respuesta;
  }

//--------------------------------------------------------------
// Devuelve un único registro consultado a la base de datos
function TedeceaBD_ConsultaRegistro ($Archivo, $Consulta)
  {
  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Ejecutamos la consulta
  $Resultado = $Conexion->query ($Consulta);

  // Nos quedamos con el primer registro
  $Respuesta = $Resultado->fetch();

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el resultado
  return $Respuesta;
  }

//--------------------------------------------------------------
// Devuelve una lista de registros consultados a la base de datos
function TedeceaBD_ConsultaListaRegistro ($Archivo, $Consulta)
  {
  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Nos quedamos con el resultado completo de la consulta
  // Nota: como no recogemos explícitamente todos los registros,
  // la base de datos puede quedar en modo "ocupada" y rechazar escrituras
  $Lista = $Conexion->query ($Consulta);

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el resultado
  return $Lista;
  }

//--------------------------------------------------------------
// Devuelve una lista de cadenas de texto como resultado de una consulta
function TedeceaBD_ConsultaListaTexto ($Archivo, $Consulta)
  {
  // La lista que responderemos
  $Lista = array();

  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Vamos leyendo cada registro y añadiéndolo a la lista
  foreach ( $Conexion->query ($Consulta) as $Registro )
    { array_push ($Lista, $Registro[0]); }

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el resultado
  return $Lista;
  }

//--------------------------------------------------------------
// Ejecuta una consulta y devuelve un texto de error
function TedeceaBD_EjecutaConsulta ($Archivo, $Consulta)
  {
  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Ejecutamos la consulta
  $Resultado = $Conexion->query ($Consulta);

  // Vemos si hay error
  $Error = $Conexion->errorInfo();

  // La respuesta depende del error
  if ( $Error[0] === '00000' )
    { $Respuesta = ""; }
  else
    { $Respuesta = $Error[2]; }

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el error
  return $Respuesta;
  }

//--------------------------------------------------------------
// Ejecuta una consulta sobre una lista de cadenas
// y devuelve un texto de error
function TedeceaBD_EjecutaConsultaListaTexto ($Archivo, $Consulta, $Lista)
  {
  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Preparamos la consulta
  $Sentencia = $Conexion->prepare ($Consulta);

  // Ejecutamos la consulta en cada cadena
  foreach ( $Lista as $Texto )
    { $Sentencia->execute (array($Texto)); }

  // Vemos si hay error
  $Error = $Conexion->errorInfo();

  // La respuesta depende del error
  if ( $Error[0] === '00000' )
    { $Respuesta = ""; }
  else
    { $Respuesta = $Error[2]; }

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el error
  return $Respuesta;
  }

//--------------------------------------------------------------
// Ejecuta una consulta sobre una lista de parámetros
// y devuelve un texto de error
function TedeceaBD_EjecutaConsultaLista ($Archivo, $Consulta, $Lista)
  {
  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Preparamos la consulta
  $Sentencia = $Conexion->prepare ($Consulta);

  // Ejecutamos la consulta en cada registro
  foreach ( $Lista as $Dato )
    { $Sentencia->execute ($Dato); }

  // Vemos si hay error
  $Error = $Conexion->errorInfo();

  // La respuesta depende del error
  if ( $Error[0] === '00000' )
    { $Respuesta = ""; }
  else
    { $Respuesta = $Error[2]; }

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el error
  return $Respuesta;
  }

//--------------------------------------------------------------
// Ejecuta una consulta de inserción, devuelve un texto de error
// y modifica una variable para devolver el identificador del
// registro creado (0 si ha fallado la inserción)
function TedeceaBD_InsertaRegistro ($Archivo, $Consulta, &$Identificador)
  {
  // Abrimos una conexión a la base de datos
  $Conexion = BD_Conexion($Archivo);

  // Ejecutamos la consulta
  $Resultado = $Conexion->query ($Consulta);

  // Vemos si hay error
  $Error = $Conexion->errorInfo();

  // La respuesta depende del error
  if ( $Error[0] === '00000' )
    {
    // Todo correcto
    $Respuesta = "";
    $Identificador = $Conexion->lastInsertId();
    }
  else
    {
    // Algo ha ido mal
    $Respuesta = $Error[2];
    $Identificador = 0;
    }

  // Cerramos la conexión
  $Conexion = null;

  // Devolvemos el error
  return $Respuesta;
  }

//----------------------------
// Funciones internas
//----------------------------

//--------------------------------------------------------------
// Atrapa las posibles excepciones
function BD_GestorExcepciones ($Excepcion)
  { echo "Excepción no capturada: " , $Excepcion->getMessage(), "\n"; }

//--------------------------------------------------------------
// Devuelve un manejador PDO para conectar con la base de datos
function BD_Conexion($FicheroBaseDato)
  {
  // Nos preparamos para manejar una excepción si no se puede
  // realizar la conexión
  set_exception_handler('BD_GestorExcepciones');

  // Abrimos una conexión a la base de datos
  $Conexion = new PDO ("sqlite:$FicheroBaseDato");

  // Devolvemos el manejador
  return $Conexion;
  }

?>
